﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/events/CloudWatchEvents_EXPORTS.h>
#include <aws/events/model/ArchiveState.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CloudWatchEvents {
namespace Model {

/**
 * <p>An <code>Archive</code> object that contains details about an
 * archive.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/Archive">AWS API
 * Reference</a></p>
 */
class Archive {
 public:
  AWS_CLOUDWATCHEVENTS_API Archive() = default;
  AWS_CLOUDWATCHEVENTS_API Archive(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHEVENTS_API Archive& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHEVENTS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the archive.</p>
   */
  inline const Aws::String& GetArchiveName() const { return m_archiveName; }
  inline bool ArchiveNameHasBeenSet() const { return m_archiveNameHasBeenSet; }
  template <typename ArchiveNameT = Aws::String>
  void SetArchiveName(ArchiveNameT&& value) {
    m_archiveNameHasBeenSet = true;
    m_archiveName = std::forward<ArchiveNameT>(value);
  }
  template <typename ArchiveNameT = Aws::String>
  Archive& WithArchiveName(ArchiveNameT&& value) {
    SetArchiveName(std::forward<ArchiveNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the event bus associated with the archive. Only events from this
   * event bus are sent to the archive.</p>
   */
  inline const Aws::String& GetEventSourceArn() const { return m_eventSourceArn; }
  inline bool EventSourceArnHasBeenSet() const { return m_eventSourceArnHasBeenSet; }
  template <typename EventSourceArnT = Aws::String>
  void SetEventSourceArn(EventSourceArnT&& value) {
    m_eventSourceArnHasBeenSet = true;
    m_eventSourceArn = std::forward<EventSourceArnT>(value);
  }
  template <typename EventSourceArnT = Aws::String>
  Archive& WithEventSourceArn(EventSourceArnT&& value) {
    SetEventSourceArn(std::forward<EventSourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current state of the archive.</p>
   */
  inline ArchiveState GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(ArchiveState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline Archive& WithState(ArchiveState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description for the reason that the archive is in the current state.</p>
   */
  inline const Aws::String& GetStateReason() const { return m_stateReason; }
  inline bool StateReasonHasBeenSet() const { return m_stateReasonHasBeenSet; }
  template <typename StateReasonT = Aws::String>
  void SetStateReason(StateReasonT&& value) {
    m_stateReasonHasBeenSet = true;
    m_stateReason = std::forward<StateReasonT>(value);
  }
  template <typename StateReasonT = Aws::String>
  Archive& WithStateReason(StateReasonT&& value) {
    SetStateReason(std::forward<StateReasonT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of days to retain events in the archive before they are
   * deleted.</p>
   */
  inline int GetRetentionDays() const { return m_retentionDays; }
  inline bool RetentionDaysHasBeenSet() const { return m_retentionDaysHasBeenSet; }
  inline void SetRetentionDays(int value) {
    m_retentionDaysHasBeenSet = true;
    m_retentionDays = value;
  }
  inline Archive& WithRetentionDays(int value) {
    SetRetentionDays(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The size of the archive, in bytes.</p>
   */
  inline long long GetSizeBytes() const { return m_sizeBytes; }
  inline bool SizeBytesHasBeenSet() const { return m_sizeBytesHasBeenSet; }
  inline void SetSizeBytes(long long value) {
    m_sizeBytesHasBeenSet = true;
    m_sizeBytes = value;
  }
  inline Archive& WithSizeBytes(long long value) {
    SetSizeBytes(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of events in the archive.</p>
   */
  inline long long GetEventCount() const { return m_eventCount; }
  inline bool EventCountHasBeenSet() const { return m_eventCountHasBeenSet; }
  inline void SetEventCount(long long value) {
    m_eventCountHasBeenSet = true;
    m_eventCount = value;
  }
  inline Archive& WithEventCount(long long value) {
    SetEventCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time stamp for the time that the archive was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  Archive& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_archiveName;
  bool m_archiveNameHasBeenSet = false;

  Aws::String m_eventSourceArn;
  bool m_eventSourceArnHasBeenSet = false;

  ArchiveState m_state{ArchiveState::NOT_SET};
  bool m_stateHasBeenSet = false;

  Aws::String m_stateReason;
  bool m_stateReasonHasBeenSet = false;

  int m_retentionDays{0};
  bool m_retentionDaysHasBeenSet = false;

  long long m_sizeBytes{0};
  bool m_sizeBytesHasBeenSet = false;

  long long m_eventCount{0};
  bool m_eventCountHasBeenSet = false;

  Aws::Utils::DateTime m_creationTime{};
  bool m_creationTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvents
}  // namespace Aws
