# This example is described in the Developer's Manual
# in the chapter "Writing A Categoriser".
#
# It will need to installed according perl convention.
package MyAnalysers::PageCategoriser;

use strict;

# A DlfCategoriser is a subclass of Lire::DlfCategoriser
use base qw/Lire::DlfCategoriser/;

# The constructor. The constructor can be called anything. It isn't
# used by the Lire framework. You'll instantiate the analyser
# yourself in the registration script.
sub new {
    return bless {}, shift;
}

########################################################################
#                          METADATA METHODS
#
# The following methods are used to provide information about your
# converter to the Lire framework.
#
########################################################################



# Returns the name of the categoriser.
sub name {
    return 'page-categoriser';
}

# A title that is available in the user interface.
sub title {
    return "A page categoriser";
}

# A DocBoook snippet giving more information about the categoriser.
sub description {
    return "<para>A categoriser that assigns categories based on a map
    of regular expressions to categories.</para>";
}

# This method should returns the name of the DLF schema for which
# records are categorised.
sub src_schema {
    return "www";
}

# This method should returns the name of the DLF schema that is 
# written to.
sub dst_schema {
    return "www-category";
}

########################################################################
#                          PROCESSING METHODS
#
# These methods are used to implement the actual behavior of your
# DLF categoriser.
#
########################################################################

# Called before the catgorisation begin. The $config
# parameter is the configuration data as entered by the
# user based on the analyser's configuration specification.
# This one should be named <name>_properties.
sub initialise {
    my ( $self, $config ) = @_;

    foreach my $map ( @$config ) {
        $map->[0] = qr/$map->[0]/;
    }

    $self->{'categories'} = $config;
    return;
}

# This is the categorising method. It will be called once for each DLF
# record of the source DLF schema that needs categorising. You assign
# the extended fields directly to the hash. Errors are reporting by
# throwing an exception (die).
sub categorise { 
    my ( $self, $dlf ) = @_;

    foreach my $map ( @{$self->{'categories'}} ) {
        if ( $dlf->{'requested_page'} =~ /$map->[0]/ ) {
            $dlf->{'category'} = $map->[1];
            return;
        }
    }
    return;
}

# Perl modules should return 1, when required.
1;
